/**
 * Marlin 3D Printer Firmware
 * Copyright (C) 2016 MarlinFirmware [https://github.com/MarlinFirmware/Marlin]
 *
 * Based on Sprinter and grbl.
 * Copyright (C) 2011 Camiel Gubbels / Erik van der Zalm
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

/**
 * Standard Marlin Status Screen bitmaps
 *
 * Use the Marlin Bitmap Converter to make your own:
 * http://marlinfw.org/tools/u8glib/converter.html
 */

#include <binary.h>

#include "../../inc/MarlinConfig.h"

#if ENABLED(CUSTOM_STATUS_SCREEN_IMAGE)

  /**
   * Custom _Statusscreen.h files can define:
   * - A custom logo image
   * - A custom heater bitmap
   * - A custom fan bitmap / animation
   *
   * See the included examples for guidance
   */
  #include "../../../_Statusscreen.h"

  #ifdef STATUS_SCREENWIDTH
    #error "Your custom _Statusscreen.h needs to be converted for Marlin 2.0."
  #endif

#endif

//
// Default Status Screen Heater or Hotends bitmaps
//

#if !STATUS_HEATERS_WIDTH && !STATUS_HOTEND1_WIDTH

  #ifdef STATUS_COMBINE_HEATERS

    //
    // Status Screen Combined Heater bitmaps
    //

    #if HAS_HEATED_BED && HOTENDS == 0
      #define STATUS_HEATERS_X 80
    #endif

    #if HAS_HEATED_BED && HOTENDS <= 3

      #if HOTENDS == 0

        #define STATUS_HEATERS_WIDTH  18

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00100000,B10000010,B00000000,
          B00010000,B01000001,B00000000,
          B00010000,B01000001,B00000000,
          B00100000,B10000010,B00000000,
          B01000001,B00000100,B00000000,
          B10000010,B00001000,B00000000,
          B10000010,B00001000,B00000000,
          B01000001,B00000100,B00000000,
          B00100000,B10000010,B00000000,
          B00000000,B00000000,B00000000,
          B11111111,B11111111,B11000000,
          B11111111,B11111111,B11000000
        };

      #elif HOTENDS == 1

        #define STATUS_HEATERS_WIDTH  90

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,
          B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,
          B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,
          B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,
          B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,
          B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,
          B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000
        };

      #elif HOTENDS == 2

        #define STATUS_HEATERS_WIDTH  90

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,
          B00111110,B11110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,
          B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,
          B00111010,B11110000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,
          B00011110,B11100000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,
          B00011110,B11100000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,
          B00111110,B11110000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,
          B00111110,B11110000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,
          B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,
          B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,
          B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000
        };

      #else // HOTENDS > 2

        #define STATUS_HEATERS_WIDTH  90

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00100000,B10000010,B00000000,
          B00111110,B11110000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00010000,B01000001,B00000000,
          B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00010000,B01000001,B00000000,
          B00111010,B11110000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00100000,B10000010,B00000000,
          B00011110,B11100000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B01000001,B00000100,B00000000,
          B00011110,B11100000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B10000010,B00001000,B00000000,
          B00111110,B11110000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B10000010,B00001000,B00000000,
          B00111110,B11110000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B01000001,B00000100,B00000000,
          B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00100000,B10000010,B00000000,
          B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B11111111,B11111111,B11000000,
          B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B11111111,B11111111,B11000000
        };

      #endif // HOTENDS

    #else // !HAS_HEATED_BED || HOTENDS > 3

      #if HOTENDS == 1

        #define STATUS_HEATERS_WIDTH  11

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00011111,B11100000,
          B00011111,B11100000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00001111,B11000000,
          B00000111,B10000000,
          B00000011,B00000000
        };

      #elif HOTENDS == 2

        #define STATUS_HEATERS_WIDTH  36

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,
          B00111110,B11110000,B00000000,B00111100,B11110000,
          B00111100,B11110000,B00000000,B00111011,B01110000,
          B00111010,B11110000,B00000000,B00111111,B01110000,
          B00011110,B11100000,B00000000,B00011110,B11100000,
          B00011110,B11100000,B00000000,B00011101,B11100000,
          B00111110,B11110000,B00000000,B00111011,B11110000,
          B00111110,B11110000,B00000000,B00111000,B01110000,
          B00111111,B11110000,B00000000,B00111111,B11110000,
          B00001111,B11000000,B00000000,B00001111,B11000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,
          B00000011,B00000000,B00000000,B00000011,B00000000
        };

      #elif HOTENDS == 3

        #define STATUS_HEATERS_WIDTH  60

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,
          B00111110,B11110000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,
          B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,
          B00111010,B11110000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,
          B00011110,B11100000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,
          B00011110,B11100000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,
          B00111110,B11110000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,
          B00111110,B11110000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,
          B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,
          B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,
          B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000
        };

      #elif HOTENDS > 3

        #define STATUS_HEATERS_WIDTH  84

        const unsigned char status_heaters_bmp[] PROGMEM = {
          B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,
          B00111110,B11110000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00111011,B01110000,
          B00111100,B11110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,
          B00111010,B11110000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00111011,B01110000,
          B00011110,B11100000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B00011011,B01100000,
          B00011110,B11100000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B00011000,B00100000,
          B00111110,B11110000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B00111111,B01110000,
          B00111110,B11110000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B00111111,B01110000,
          B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,
          B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,
          B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,
          B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000
        };

      #endif // HOTENDS

    #endif // !HAS_HEATED_BED || HOTENDS > 3

  #else // !STATUS_COMBINE_HEATERS

    //
    // Status Screen Hotends bitmaps
    //

    #if HOTENDS

      #define STATUS_HOTEND1_WIDTH  12

      #if HOTENDS == 1 || defined(STATUS_HOTEND_NUMBERLESS)

        const unsigned char status_hotend_a_bmp[] PROGMEM = {
          B00011111,B11100000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00011111,B11100000,
          B00011111,B11100000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00111111,B11110000,
          B00001111,B11000000,
          B00000111,B10000000,
          B00000011,B00000000
        };

        #ifdef STATUS_HOTEND_ANIM

          const unsigned char status_hotend_b_bmp[] PROGMEM = {
            B00011111,B11100000,
            B00100000,B00010000,
            B00100000,B00010000,
            B00100000,B00010000,
            B00010000,B00100000,
            B00010000,B00100000,
            B00100000,B00010000,
            B00100000,B00010000,
            B00110000,B00110000,
            B00001000,B01000000,
            B00000100,B10000000,
            B00000011,B00000000
          };

        #endif

      #else

        #if HOTENDS >= 2

          #if HAS_HEATED_BED
            #define MAX_HOTEND_BITMAPS 3
          #else
            #define MAX_HOTEND_BITMAPS 4
          #endif
          #if HOTENDS > MAX_HOTEND_BITMAPS
            #define STATUS_HOTEND_BITMAPS MAX_HOTEND_BITMAPS
          #else
            #define STATUS_HOTEND_BITMAPS HOTENDS
          #endif

          #ifdef STATUS_HOTEND_ANIM

            const unsigned char status_hotend1_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111111,B11110000,
              B00111110,B11110000,
              B00111100,B11110000,
              B00011010,B11100000,
              B00011110,B11100000,
              B00111110,B11110000,
              B00111110,B11110000,
              B00111110,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

            const unsigned char status_hotend1_b_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00100000,B00010000,
              B00100001,B00010000,
              B00100011,B00010000,
              B00010101,B00100000,
              B00010001,B00100000,
              B00100001,B00010000,
              B00100001,B00010000,
              B00110001,B00110000,
              B00001000,B01000000,
              B00000100,B10000000,
              B00000011,B00000000
            };

            const unsigned char status_hotend2_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111111,B11110000,
              B00111100,B11110000,
              B00111011,B01110000,
              B00011111,B01100000,
              B00011110,B11100000,
              B00111101,B11110000,
              B00111011,B11110000,
              B00111000,B01110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

            const unsigned char status_hotend2_b_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00100000,B00010000,
              B00100011,B00010000,
              B00100100,B10010000,
              B00010000,B10100000,
              B00010001,B00100000,
              B00100010,B00010000,
              B00100100,B00010000,
              B00110111,B10110000,
              B00001000,B01000000,
              B00000100,B10000000,
              B00000011,B00000000
            };

          #else

            const unsigned char status_hotend1_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111110,B11110000,
              B00111100,B11110000,
              B00111010,B11110000,
              B00011110,B11100000,
              B00011110,B11100000,
              B00111110,B11110000,
              B00111110,B11110000,
              B00111111,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

            const unsigned char status_hotend2_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111100,B11110000,
              B00111011,B01110000,
              B00111111,B01110000,
              B00011110,B11100000,
              B00011101,B11100000,
              B00111011,B11110000,
              B00111000,B01110000,
              B00111111,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

          #endif

        #endif

        #if STATUS_HOTEND_BITMAPS >= 3

          #ifdef STATUS_HOTEND_ANIM

            const unsigned char status_hotend3_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111111,B11110000,
              B00111100,B11110000,
              B00111011,B01110000,
              B00011111,B01100000,
              B00011100,B11100000,
              B00111111,B01110000,
              B00111011,B01110000,
              B00111100,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

            const unsigned char status_hotend3_b_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00100000,B00010000,
              B00100011,B00010000,
              B00100100,B10010000,
              B00010000,B10100000,
              B00010011,B00100000,
              B00100000,B10010000,
              B00100100,B10010000,
              B00110011,B00110000,
              B00001000,B01000000,
              B00000100,B10000000,
              B00000011,B00000000
            };

          #else

            const unsigned char status_hotend3_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111100,B11110000,
              B00111011,B01110000,
              B00111111,B01110000,
              B00011100,B11100000,
              B00011111,B01100000,
              B00111011,B01110000,
              B00111100,B11110000,
              B00111111,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

          #endif

        #endif

        #if STATUS_HOTEND_BITMAPS >= 4 && !HAS_HEATED_BED

          #ifdef STATUS_HOTEND_ANIM

            const unsigned char status_hotend4_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111111,B11110000,
              B00111011,B01110000,
              B00111011,B01110000,
              B00011011,B01100000,
              B00011011,B01100000,
              B00111000,B00110000,
              B00111111,B01110000,
              B00111111,B01110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

            const unsigned char status_hotend4_b_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00100000,B00010000,
              B00100100,B10010000,
              B00100100,B10010000,
              B00010100,B10100000,
              B00010100,B10100000,
              B00100111,B11010000,
              B00100000,B10010000,
              B00110000,B10110000,
              B00001000,B01000000,
              B00000100,B10000000,
              B00000011,B00000000
            };

          #else

            const unsigned char status_hotend4_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111011,B01110000,
              B00111011,B01110000,
              B00111011,B01110000,
              B00011011,B01100000,
              B00011000,B00100000,
              B00111111,B01110000,
              B00111111,B01110000,
              B00111111,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

          #endif

        #endif

        #if STATUS_HOTEND_BITMAPS >= 5 && !HAS_HEATED_BED

          #ifdef STATUS_HOTEND_ANIM

            const unsigned char status_hotend5_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111111,B11110000,
              B00111000,B01110000,
              B00111011,B11110000,
              B00011000,B11100000,
              B00011111,B01100000,
              B00111111,B01110000,
              B00111011,B01110000,
              B00111100,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

            const unsigned char status_hotend5_b_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00100000,B00010000,
              B00100111,B10010000,
              B00100100,B00010000,
              B00010111,B00100000,
              B00010000,B10100000,
              B00100000,B10010000,
              B00100100,B10010000,
              B00110011,B00110000,
              B00001000,B01000000,
              B00000100,B10000000,
              B00000011,B00000000
            };

          #else

            const unsigned char status_hotend5_a_bmp[] PROGMEM = {
              B00011111,B11100000,
              B00111000,B01110000,
              B00111011,B11110000,
              B00111000,B11110000,
              B00011111,B01100000,
              B00011111,B01100000,
              B00111011,B01110000,
              B00111100,B11110000,
              B00111111,B11110000,
              B00001111,B11000000,
              B00000111,B10000000,
              B00000011,B00000000
            };

          #endif

        #endif

      #endif

    #endif

  #endif

#endif // !STATUS_HEATERS_WIDTH && !STATUS_HOTEND1_WIDTH

//
// Default Status Screen Bed bitmaps
//

#if !STATUS_BED_WIDTH && !defined(STATUS_COMBINE_HEATERS) && HAS_HEATED_BED && HOTENDS < 4

  #ifdef STATUS_ALT_BED_BITMAP

    #define STATUS_BED_ANIM
    #define STATUS_BED_WIDTH  24
    #ifndef STATUS_BED_X
      #define STATUS_BED_X    72
    #endif
    #define STATUS_BED_TEXT_X STATUS_BED_X + 13

    const unsigned char status_bed_bmp[] PROGMEM = {
      B11111111,B11111111,B11000000,
      B01000000,B00000000,B00100000,
      B00100000,B00000000,B00010000,
      B00010000,B00000000,B00001000,
      B00001000,B00000000,B00000100,
      B00000100,B00000000,B00000010,
      B00000011,B11111111,B11111111
    };

    const unsigned char status_bed_on_bmp[] PROGMEM = {
      B00000010,B00100010,B00000000,
      B00000100,B01000100,B00000000,
      B00000100,B01000100,B00000000,
      B00000010,B00100010,B00000000,
      B00000001,B00010001,B00000000,
      B11111111,B11111111,B11000000,
      B01000000,B10001000,B10100000,
      B00100001,B00010001,B00010000,
      B00010010,B00100010,B00001000,
      B00001000,B00000000,B00000100,
      B00000100,B00000000,B00000010,
      B00000011,B11111111,B11111111
    };

  #else

    #define STATUS_BED_WIDTH  18
    #ifndef STATUS_BED_X
      #define STATUS_BED_X    80
    #endif

    #ifdef STATUS_BED_ANIM

      const unsigned char status_bed_bmp[] PROGMEM = {
        B11111111,B11111111,B11000000,
        B11111111,B11111111,B11000000
      };

      const unsigned char status_bed_on_bmp[] PROGMEM = {
        B00100000,B10000010,B00000000,
        B00010000,B01000001,B00000000,
        B00010000,B01000001,B00000000,
        B00100000,B10000010,B00000000,
        B01000001,B00000100,B00000000,
        B10000010,B00001000,B00000000,
        B10000010,B00001000,B00000000,
        B01000001,B00000100,B00000000,
        B00100000,B10000010,B00000000,
        B00000000,B00000000,B00000000,
        B11111111,B11111111,B11000000,
        B11111111,B11111111,B11000000
      };

    #else

      const unsigned char status_bed_bmp[] PROGMEM = {
        B00100000,B10000010,B00000000,
        B00010000,B01000001,B00000000,
        B00010000,B01000001,B00000000,
        B00100000,B10000010,B00000000,
        B01000001,B00000100,B00000000,
        B10000010,B00001000,B00000000,
        B10000010,B00001000,B00000000,
        B01000001,B00000100,B00000000,
        B00100000,B10000010,B00000000,
        B00000000,B00000000,B00000000,
        B11111111,B11111111,B11000000,
        B11111111,B11111111,B11000000
      };

    #endif

  #endif

#endif // !STATUS_BED_WIDTH && !STATUS_COMBINE_HEATERS && HAS_HEATED_BED && HOTENDS < 4

// Can also be overridden in Configuration.h
// If you can afford it, try the 3-frame fan animation!
// Don't compile in the fan animation with no fan
#if !HAS_FAN0
  #undef STATUS_FAN_FRAMES
#elif !defined(STATUS_FAN_FRAMES)
  #define STATUS_FAN_FRAMES 2
#elif STATUS_FAN_FRAMES > 3
  #error "Only 3 fan animation frames currently supported."
#endif

//
// Provide default Fan Bitmaps
//
#if !defined(STATUS_FAN_WIDTH) && STATUS_FAN_FRAMES > 0

  // Provide a fan animation if none exists

  #if STATUS_FAN_FRAMES <= 2

    #define STATUS_FAN_Y      2
    #define STATUS_FAN_WIDTH 20

    const unsigned char status_fan0_bmp[] PROGMEM = {
      B00111111,B11111111,B11110000,
      B00111000,B00000000,B01110000,
      B00110000,B11111100,B00110000,
      B00100000,B11111100,B00010000,
      B00100000,B01111000,B00010000,
      B00100000,B00110000,B00010000,
      B00101100,B00000000,B11010000,
      B00101110,B00110001,B11010000,
      B00101111,B01111011,B11010000,
      B00101111,B01111011,B11010000,
      B00101110,B00110001,B11010000,
      B00101100,B00000000,B11010000,
      B00100000,B00110000,B00010000,
      B00100000,B01111000,B00010000,
      B00100000,B11111100,B00010000,
      B00110000,B11111100,B00110000,
      B00111000,B00000000,B01110000,
      B00111111,B11111111,B11110000
    };

    #if STATUS_FAN_FRAMES == 2
      const unsigned char status_fan1_bmp[] PROGMEM = {
        B00111111,B11111111,B11110000,
        B00111000,B00000000,B01110000,
        B00110001,B10000110,B00110000,
        B00100011,B10000111,B00010000,
        B00100111,B10000111,B10010000,
        B00101111,B10000111,B11010000,
        B00101111,B00000011,B11010000,
        B00100000,B00110000,B00010000,
        B00100000,B01111000,B00010000,
        B00100000,B01111000,B00010000,
        B00100000,B00110000,B00010000,
        B00101111,B00000011,B11010000,
        B00101111,B10000111,B11010000,
        B00100111,B10000111,B10010000,
        B00100011,B10000111,B00010000,
        B00110001,B10000110,B00110000,
        B00111000,B00000000,B01110000,
        B00111111,B11111111,B11110000
      };
    #endif

  #elif STATUS_FAN_FRAMES == 3

    #define STATUS_FAN_WIDTH 21

    const unsigned char status_fan0_bmp[] PROGMEM = {
      B00111111,B11111111,B11111000,
      B00111110,B00000000,B11111000,
      B00111001,B00000001,B00111000,
      B00110111,B10000011,B11011000,
      B00110111,B10000011,B11011000,
      B00101111,B11000111,B11101000,
      B00100111,B11000111,B11001000,
      B00100001,B11111111,B00001000,
      B00100000,B01111100,B00001000,
      B00100000,B01111100,B00001000,
      B00100000,B01111100,B00001000,
      B00100001,B11111111,B00001000,
      B00100111,B11000111,B11001000,
      B00101111,B11000111,B11101000,
      B00110111,B10000011,B11011000,
      B00110111,B10000011,B11011000,
      B00111001,B00000001,B00111000,
      B00111110,B00000000,B11111000,
      B00111111,B11111111,B11111000
    };
    const unsigned char status_fan1_bmp[] PROGMEM = {
      B00111111,B11111111,B11111000,
      B00111110,B00110000,B11111000,
      B00111001,B11110000,B00111000,
      B00110001,B11110000,B00011000,
      B00110000,B11110000,B00011000,
      B00100000,B11110000,B01101000,
      B00100000,B00110001,B11101000,
      B00100000,B00111001,B11101000,
      B00100000,B01111111,B11111000,
      B00111111,B11111111,B11111000,
      B00111111,B11111100,B00001000,
      B00101111,B00111000,B00001000,
      B00101110,B00011000,B00001000,
      B00101100,B00011110,B00001000,
      B00110000,B00011110,B00011000,
      B00110000,B00011111,B00011000,
      B00111000,B00011111,B00111000,
      B00111110,B00011000,B11111000,
      B00111111,B11111111,B11111000
    };
    const unsigned char status_fan2_bmp[] PROGMEM = {
      B00111111,B11111111,B11111000,
      B00111110,B00011000,B11111000,
      B00111000,B00011111,B00111000,
      B00110000,B00011111,B10011000,
      B00110100,B00011111,B00011000,
      B00101110,B00011110,B00001000,
      B00101111,B00011100,B00001000,
      B00101111,B10111000,B00001000,
      B00111111,B11111100,B00001000,
      B00111111,B11111111,B11111000,
      B00100000,B01111111,B11111000,
      B00100000,B00111011,B11101000,
      B00100000,B01110001,B11101000,
      B00100000,B11110000,B11101000,
      B00110001,B11110000,B01011000,
      B00110011,B11110000,B00011000,
      B00111001,B11110000,B00111000,
      B00111110,B00110000,B11111000,
      B00111111,B11111111,B11111000
    };

  #elif STATUS_FAN_FRAMES == 4

    #define STATUS_FAN_WIDTH 21

    const unsigned char status_fan0_bmp[] PROGMEM = {
      B00111111,B11111111,B11111000,
      B00111110,B00000000,B11111000,
      B00111000,B00111111,B00111000,
      B00110000,B01111110,B00011000,
      B00110000,B01111100,B00011000,
      B00101000,B01111100,B00001000,
      B00101100,B00111000,B00001000,
      B00101111,B00111001,B11001000,
      B00101111,B11111111,B11101000,
      B00101111,B11000111,B11101000,
      B00101111,B11111111,B11101000,
      B00100111,B00111001,B11101000,
      B00100000,B00111000,B01101000,
      B00100000,B01111100,B00101000,
      B00110000,B01111100,B00011000,
      B00110000,B11111100,B00011000,
      B00111001,B11111000,B00111000,
      B00111110,B00000000,B11111000,
      B00111111,B11111111,B11111000
    };
    const unsigned char status_fan1_bmp[] PROGMEM = {
      B00111111,B11111111,B11111000,
      B00111110,B00000000,B11111000,
      B00111000,B00001111,B00111000,
      B00110100,B00011111,B11011000,
      B00110110,B00011111,B10011000,
      B00101111,B00011111,B00001000,
      B00101111,B10011110,B00001000,
      B00101111,B11111100,B00001000,
      B00101111,B11011100,B00001000,
      B00100111,B11101111,B11001000,
      B00100000,B01110111,B11101000,
      B00100000,B01111111,B11101000,
      B00100000,B11110011,B11101000,
      B00100001,B11110001,B11101000,
      B00110011,B11110000,B11011000,
      B00110111,B11110000,B01011000,
      B00111001,B11100000,B00111000,
      B00111110,B00000000,B11111000,
      B00111111,B11111111,B11111000
    };
    const unsigned char status_fan2_bmp[] PROGMEM = {
      B00111111,B11111111,B11111000,
      B00111110,B10000000,B11111000,
      B00111001,B10000000,B00111000,
      B00110111,B10000001,B11011000,
      B00110111,B11000011,B11011000,
      B00100111,B11000111,B11101000,
      B00100011,B11000111,B11111000,
      B00100001,B11111111,B10001000,
      B00100000,B01101100,B00001000,
      B00100000,B01101100,B00001000,
      B00100000,B01101100,B00001000,
      B00100011,B11111111,B00001000,
      B00111111,B11000111,B10001000,
      B00101111,B11000111,B11001000,
      B00110111,B10000111,B11011000,
      B00110111,B00000011,B11011000,
      B00111000,B00000011,B00111000,
      B00111110,B00000010,B11111000,
      B00111111,B11111111,B11111000
    };
    const unsigned char status_fan3_bmp[] PROGMEM = {
      B00111111,B11111111,B11111000,
      B00111110,B00000000,B11111000,
      B00111001,B11110000,B00111000,
      B00110001,B11100000,B00011000,
      B00110001,B11100000,B00011000,
      B00100001,B11100001,B11101000,
      B00100000,B11110011,B11101000,
      B00100000,B01111111,B11101000,
      B00100000,B01110111,B11101000,
      B00101000,B11101110,B00101000,
      B00101111,B11011100,B00001000,
      B00101111,B11111100,B00001000,
      B00101111,B10011110,B00001000,
      B00101111,B00001111,B00001000,
      B00110000,B00001111,B00011000,
      B00110000,B00001111,B00011000,
      B00111000,B00011111,B00111000,
      B00111110,B00000000,B11111000,
      B00111111,B11111111,B11111000
    };

  #endif

#endif

//
// Custom Logo Bitmap Properties
//
#ifndef STATUS_LOGO_WIDTH
  #define STATUS_LOGO_WIDTH 0
#endif
#ifndef STATUS_LOGO_BYTEWIDTH
  #define STATUS_LOGO_BYTEWIDTH ((STATUS_LOGO_WIDTH + 7) / 8)
#endif
#if STATUS_LOGO_WIDTH
  #ifndef STATUS_LOGO_X
    #define STATUS_LOGO_X 0
  #endif
  #ifndef STATUS_LOGO_Y
    #define STATUS_LOGO_Y MIN(0, 10 - (STATUS_LOGO_HEIGHT) / 2)
  #endif
  #ifndef STATUS_LOGO_HEIGHT
    #define STATUS_LOGO_HEIGHT (sizeof(status_logo_bmp) / (STATUS_LOGO_BYTEWIDTH))
  #endif
  static_assert(
    sizeof(status_logo_bmp) == (STATUS_LOGO_BYTEWIDTH) * (STATUS_LOGO_HEIGHT),
    "Status logo bitmap (status_logo_bmp) dimensions don't match data."
  );
#endif

//
// Heater Bitmap Properties
//
#if STATUS_HOTEND1_WIDTH || STATUS_HEATERS_WIDTH

  #ifndef STATUS_HEATERS_XSPACE
    #define STATUS_HEATERS_XSPACE 24 // Like the included bitmaps
  #endif

  #ifndef STATUS_HEATERS_X
    #if STATUS_LOGO_BYTEWIDTH
      #define STATUS_HEATERS_X ((STATUS_LOGO_BYTEWIDTH + 0) * 8)
    #else
      #define STATUS_HEATERS_X 8 // Like the included bitmaps
    #endif
  #endif

#endif

#if STATUS_HOTEND1_WIDTH

  //
  // Hotend images. A base hotend image and optional "ON" state image.
  //

  #ifndef STATUS_HOTEND_BITMAPS
    #define STATUS_HOTEND_BITMAPS 1
  #endif

  #ifndef STATUS_HOTEND2_WIDTH
    #define STATUS_HOTEND2_WIDTH STATUS_HOTEND1_WIDTH
  #endif
  #ifndef STATUS_HOTEND3_WIDTH
    #define STATUS_HOTEND3_WIDTH STATUS_HOTEND2_WIDTH
  #endif
  #ifndef STATUS_HOTEND4_WIDTH
    #define STATUS_HOTEND4_WIDTH STATUS_HOTEND3_WIDTH
  #endif
  #ifndef STATUS_HOTEND5_WIDTH
    #define STATUS_HOTEND5_WIDTH STATUS_HOTEND5_WIDTH
  #endif
  #ifndef STATUS_HOTEND6_WIDTH
    #define STATUS_HOTEND6_WIDTH STATUS_HOTEND6_WIDTH
  #endif

  constexpr uint8_t status_hotend_width[HOTENDS] = ARRAY_N(HOTENDS, STATUS_HOTEND1_WIDTH, STATUS_HOTEND2_WIDTH, STATUS_HOTEND3_WIDTH, STATUS_HOTEND4_WIDTH, STATUS_HOTEND5_WIDTH, STATUS_HOTEND6_WIDTH);
  #define STATUS_HOTEND_WIDTH(N) status_hotend_width[N]

  #ifndef STATUS_HOTEND1_BYTEWIDTH
    #define STATUS_HOTEND1_BYTEWIDTH ((STATUS_HOTEND1_WIDTH + 7) / 8)
  #endif
  #ifndef STATUS_HOTEND2_BYTEWIDTH
    #define STATUS_HOTEND2_BYTEWIDTH ((STATUS_HOTEND2_WIDTH + 7) / 8)
  #endif
  #ifndef STATUS_HOTEND3_BYTEWIDTH
    #define STATUS_HOTEND3_BYTEWIDTH ((STATUS_HOTEND3_WIDTH + 7) / 8)
  #endif
  #ifndef STATUS_HOTEND4_BYTEWIDTH
    #define STATUS_HOTEND4_BYTEWIDTH ((STATUS_HOTEND4_WIDTH + 7) / 8)
  #endif
  #ifndef STATUS_HOTEND5_BYTEWIDTH
    #define STATUS_HOTEND5_BYTEWIDTH ((STATUS_HOTEND5_WIDTH + 7) / 8)
  #endif
  #ifndef STATUS_HOTEND6_BYTEWIDTH
    #define STATUS_HOTEND6_BYTEWIDTH ((STATUS_HOTEND6_WIDTH + 7) / 8)
  #endif

  constexpr uint8_t status_hotend_bytewidth[HOTENDS] = ARRAY_N(HOTENDS, STATUS_HOTEND1_BYTEWIDTH, STATUS_HOTEND2_BYTEWIDTH, STATUS_HOTEND3_BYTEWIDTH, STATUS_HOTEND4_BYTEWIDTH, STATUS_HOTEND5_BYTEWIDTH, STATUS_HOTEND6_BYTEWIDTH);
  #define STATUS_HOTEND_BYTEWIDTH(N) status_hotend_bytewidth[N]

  #ifndef STATUS_HOTEND1_X
    #define STATUS_HOTEND1_X STATUS_HEATERS_X
  #endif
  #ifndef STATUS_HOTEND2_X
    #define STATUS_HOTEND2_X STATUS_HOTEND1_X + STATUS_HEATERS_XSPACE
  #endif
  #ifndef STATUS_HOTEND3_X
    #define STATUS_HOTEND3_X STATUS_HOTEND2_X + STATUS_HEATERS_XSPACE
  #endif
  #ifndef STATUS_HOTEND4_X
    #define STATUS_HOTEND4_X STATUS_HOTEND3_X + STATUS_HEATERS_XSPACE
  #endif
  #ifndef STATUS_HOTEND5_X
    #define STATUS_HOTEND5_X STATUS_HOTEND5_X + STATUS_HEATERS_XSPACE
  #endif
  #ifndef STATUS_HOTEND6_X
    #define STATUS_HOTEND6_X STATUS_HOTEND6_X + STATUS_HEATERS_XSPACE
  #endif

  #if HOTENDS > 2
    constexpr uint8_t status_hotend_x[HOTENDS] = ARRAY_N(HOTENDS, STATUS_HOTEND1_X, STATUS_HOTEND2_X, STATUS_HOTEND3_X, STATUS_HOTEND4_X, STATUS_HOTEND5_X, STATUS_HOTEND6_X);
    #define STATUS_HOTEND_X(N) status_hotend_x[N]
  #elif HOTENDS > 1
    #define STATUS_HOTEND_X(N) ((N) ? STATUS_HOTEND2_X : STATUS_HOTEND1_X)
  #else
    #define STATUS_HOTEND_X(N) STATUS_HOTEND1_X
  #endif

  #ifndef STATUS_HOTEND_TEXT_X
    #ifdef STATUS_HOTEND1_TEXT_X
      #ifndef STATUS_HOTEND2_TEXT_X
        #define STATUS_HOTEND2_TEXT_X STATUS_HOTEND1_TEXT_X + STATUS_HEATERS_XSPACE
      #endif
      #ifndef STATUS_HOTEND3_TEXT_X
        #define STATUS_HOTEND3_TEXT_X STATUS_HOTEND2_TEXT_X + STATUS_HEATERS_XSPACE
      #endif
      #ifndef STATUS_HOTEND4_TEXT_X
        #define STATUS_HOTEND4_TEXT_X STATUS_HOTEND3_TEXT_X + STATUS_HEATERS_XSPACE
      #endif
      #ifndef STATUS_HOTEND5_TEXT_X
        #define STATUS_HOTEND5_TEXT_X STATUS_HOTEND5_TEXT_X + STATUS_HEATERS_XSPACE
      #endif
      #ifndef STATUS_HOTEND6_TEXT_X
        #define STATUS_HOTEND6_TEXT_X STATUS_HOTEND6_TEXT_X + STATUS_HEATERS_XSPACE
      #endif
      constexpr uint8_t status_hotend_text_x[] = ARRAY_N(HOTENDS, STATUS_HOTEND1_TEXT_X, STATUS_HOTEND2_TEXT_X, STATUS_HOTEND3_TEXT_X, STATUS_HOTEND4_TEXT_X, STATUS_HOTEND5_TEXT_X, STATUS_HOTEND6_TEXT_X);
      #define STATUS_HOTEND_TEXT_X(N) status_hotend_text_x[N]
    #else
      #define STATUS_HOTEND_TEXT_X(N) (STATUS_HOTEND1_X + 6 + (N) * (STATUS_HEATERS_XSPACE))
    #endif
  #endif

  #if STATUS_HOTEND_BITMAPS > 1
    #define TEST_BITMAP_OFF status_hotend1_a_bmp
    #define TEST_BITMAP_ON  status_hotend1_b_bmp
  #else
    #define TEST_BITMAP_OFF status_hotend_a_bmp
    #define TEST_BITMAP_ON  status_hotend_b_bmp
  #endif
  #ifndef STATUS_HEATERS_HEIGHT
    #define STATUS_HEATERS_HEIGHT (sizeof(TEST_BITMAP_OFF) / (STATUS_HOTEND1_BYTEWIDTH))
  #endif
  #ifndef STATUS_HEATERS_Y
    #define STATUS_HEATERS_Y (20 - (STATUS_HEATERS_HEIGHT))
  #endif

  #define HOTEND0_BITMAP_SIZE (STATUS_HOTEND1_BYTEWIDTH) * (STATUS_HEATERS_HEIGHT)
  static_assert(
    sizeof(TEST_BITMAP_OFF) == HOTEND0_BITMAP_SIZE,
    "Status hotend bitmap (" STRINGIFY(TEST_BITMAP_OFF) ") dimensions don't match data."
  );
  #ifdef STATUS_HOTEND_ANIM
    static_assert(
      sizeof(TEST_BITMAP_ON) == HOTEND0_BITMAP_SIZE,
      "Status hotend bitmaps (" STRINGIFY(TEST_BITMAP_OFF) " and " STRINGIFY(TEST_BITMAP_ON) ") dimensions don't match."
    );
  #endif

#elif STATUS_HEATERS_WIDTH

  #ifndef STATUS_HOTEND_TEXT_X
    #define STATUS_HOTEND_TEXT_X(N) (STATUS_HEATERS_X + 6 + (N) * (STATUS_HEATERS_XSPACE))
  #endif
  #ifndef STATUS_HEATERS_BYTEWIDTH
    #define STATUS_HEATERS_BYTEWIDTH ((STATUS_HEATERS_WIDTH + 7) / 8)
  #endif
  #ifndef STATUS_HEATERS_HEIGHT
    #define STATUS_HEATERS_HEIGHT (sizeof(status_heaters_bmp) / (STATUS_HEATERS_BYTEWIDTH))
  #endif
  #ifndef STATUS_HEATERS_Y
    #define STATUS_HEATERS_Y (20 - (STATUS_HEATERS_HEIGHT))
  #endif

  static_assert(
    sizeof(status_heaters_bmp) == (STATUS_HEATERS_BYTEWIDTH) * (STATUS_HEATERS_HEIGHT),
    "Status heaters bitmap (status_heaters_bmp) dimensions don't match data."
  );

#endif

//
// Bed Bitmap Properties
//
#ifndef STATUS_BED_WIDTH
  #define STATUS_BED_WIDTH 0
#endif
#ifndef STATUS_BED_BYTEWIDTH
  #define STATUS_BED_BYTEWIDTH ((STATUS_BED_WIDTH + 7) / 8)
#endif
#if STATUS_BED_WIDTH

  #ifndef STATUS_BED_X
    #define STATUS_BED_X (128 - (STATUS_FAN_BYTEWIDTH + STATUS_BED_BYTEWIDTH) * 8)
  #endif

  #ifndef STATUS_BED_HEIGHT
    #ifdef STATUS_BED_ANIM
      #define STATUS_BED_HEIGHT(S) ((S) ? sizeof(status_bed_on_bmp) / (STATUS_BED_BYTEWIDTH) : sizeof(status_bed_bmp) / (STATUS_BED_BYTEWIDTH))
    #else
      #define STATUS_BED_HEIGHT(S) (sizeof(status_bed_bmp) / (STATUS_BED_BYTEWIDTH))
    #endif
  #endif

  #ifndef STATUS_BED_Y
    #define STATUS_BED_Y(S) (20 - STATUS_BED_HEIGHT(S))
  #endif

  #ifndef STATUS_BED_TEXT_X
    #define STATUS_BED_TEXT_X (STATUS_BED_X + 8)
  #endif

  static_assert(
    sizeof(status_bed_bmp) == (STATUS_BED_BYTEWIDTH) * (STATUS_BED_HEIGHT(0)),
    "Status bed bitmap (status_bed_bmp) dimensions don't match data."
  );
  #ifdef STATUS_BED_ANIM
    static_assert(
      sizeof(status_bed_on_bmp) == (STATUS_BED_BYTEWIDTH) * (STATUS_BED_HEIGHT(1)),
      "Status bed bitmap (status_bed_on_bmp) dimensions don't match data."
    );
  #endif

#endif

//
// Fan Bitmap Properties
//
#ifndef STATUS_FAN_WIDTH
  #define STATUS_FAN_WIDTH 0
#endif
#ifndef STATUS_FAN_BYTEWIDTH
  #define STATUS_FAN_BYTEWIDTH ((STATUS_FAN_WIDTH + 7) / 8)
#endif
#if STATUS_FAN_FRAMES
  #ifndef STATUS_FAN_X
    #define STATUS_FAN_X (128 - (STATUS_FAN_BYTEWIDTH) * 8)
  #endif
  #ifndef STATUS_FAN_Y
    #define STATUS_FAN_Y 1
  #endif
  #ifndef STATUS_FAN_TEXT_X
    #define STATUS_FAN_TEXT_X 103
  #endif
  #ifndef STATUS_FAN_TEXT_Y
    #define STATUS_FAN_TEXT_Y 28
  #endif
  #ifndef STATUS_FAN_HEIGHT
    #define STATUS_FAN_HEIGHT (sizeof(status_fan0_bmp) / (STATUS_FAN_BYTEWIDTH))
  #endif
  #define FAN_BMP_SIZE (STATUS_FAN_BYTEWIDTH) * (STATUS_FAN_HEIGHT)
  static_assert(sizeof(status_fan0_bmp) == FAN_BMP_SIZE, "Status fan bitmap (status_fan0_bmp) dimensions don't match data.");
  #if STATUS_FAN_FRAMES > 1
    static_assert(sizeof(status_fan1_bmp) == FAN_BMP_SIZE, "Status fan bitmap (status_fan1_bmp) dimensions don't match data.");
    #if STATUS_FAN_FRAMES > 2
      static_assert(sizeof(status_fan2_bmp) == FAN_BMP_SIZE, "Status fan bitmap (status_fan2_bmp) dimensions don't match data.");
      #if STATUS_FAN_FRAMES > 3
        static_assert(sizeof(status_fan3_bmp) == FAN_BMP_SIZE, "Status fan bitmap (status_fan3_bmp) dimensions don't match data.");
      #endif
    #endif
  #endif
#endif
